
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Sun March 22 10:17:10 2009
   *  Last modified   :  %modify_time%
   */
  /** @file 
   *  \brief This file includes the KMNG activate and use symmetric keys 
   *          ATP tests						
   *
   *  \version KMNG_SEP_ATP_ActivateSymKeys.c#1:csrc:1
   *  \author Annag
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ****************/
#include <stdio.h>
#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_KMNG.h"
#include "CRYS_AES.h"
#include "CRYS_DES.h"
#include "CRYS_error.h"
#include "CRYS_HASH.h"
#include "CRYS_HMAC.h"
#include "CRYS_RSA_Types.h"
#include "KMNG_Defs.h"
#include "KMNG_SEP_ATP_Defines.h"
#include "SEP_ATP_UTIL_funcs.h"
#include "KMNG_SEP_ATP_ImportExportKeys.h"
#include "KMNG_SEP_ATP_ActivateSymKeys.h"

#include "KMNG_Defs.h"
#include "LLF_KMNG.h"
#include "KMNG_SEP_ATP_TestDataDefines.h"



/************************ Defines *******************************/
/************************ Enums *********************************/
/************************ Typedefs ******************************/
/************************ Global Data ***************************/

/* pointer to the struct with symmetric key test data vector */
extern TST_Ext_KeysSym_TestDataStruct *SEP_KeysOpSym_TestData;


/************* Private function prototype ***********************/

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_HMACIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_HMACWrappedKey - wrapped HMAC key
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function execute KMNG HMAC integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_HMAC operation
* 2. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_HMACIntegrated(DxUint32_t              TST_index,
										 KMNG_HMAC_WrappedKey_t	 TST_HMACWrappedKey,
										 DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*			TST_Input_ptr;
	DxError_t			TST_Status = DX_SUCCESS;
	CRYS_HASH_Result_t  TST_Output; 

	TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];
	
	/* 1. Execute CRYS_KMNG_HMAC operation */
	TST_Status = CRYS_KMNG_HMAC(TST_HMACWrappedKey,
		SEP_KeysOpSym_TestData[TST_index].TST_HASHMode,       
		TST_Input_ptr,        
		SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
		TST_Output);   

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_HMAC",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* 2. Verify that output is as expected */
	TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,TST_Output,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);
	
	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);
EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_HMACNonIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_HMACWrappedKey - wrapped HMAC key
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function executes KMNG HMAC non integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_HMAC_Init operation
* 2. Execute CRYS_HMAC_Update operation
* 3. Execute CRYS_HMAC_Finish operation
* 4. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_HMACNonIntegrated(DxUint32_t              TST_index,
										     KMNG_HMAC_WrappedKey_t	 TST_HMACWrappedKey,
										     DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*				TST_Input_ptr;
	DxError_t				TST_Status = DX_SUCCESS;
	CRYS_HASH_Result_t		TST_Output; 
	CRYS_HMACUserContext_t  TST_HMACContext;  

	TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];

	/* 1. Execute CRYS_KMNG_HMAC_Init operation */
	/***************************************/
	TST_Status = CRYS_KMNG_HMAC_Init(TST_HMACWrappedKey,
									 &TST_HMACContext,
									 SEP_KeysOpSym_TestData[TST_index].TST_HASHMode);   

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_HMAC_Init",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 2. Execute CRYS_HMAC_Update operation */
	/***************************************/
	TST_Status = CRYS_HMAC_Update(&TST_HMACContext,   
								  TST_Input_ptr,     
								  SEP_KeysOpSym_TestData[TST_index].TST_DataLen); 
	
	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_HMAC_Update",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 3. Execute CRYS_HMAC_Finish operation */
	/***************************************/
	TST_Status = CRYS_HMAC_Finish(&TST_HMACContext ,
								  TST_Output);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
			"CRYS_HMAC_Finish",
			SEP_KeysOpSym_TestData[TST_index].TST_Name,
			TST_MasterTestName);

	/* 4. Verify that output is as expected */
	/****************************************/
	TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,TST_Output,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_DESIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_DESWrappedKey - wrapped DES key
*	TST_OpMode - operation mode
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function execute KMNG DES integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_DES operation
* 2. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_DESIntegrated(DxUint32_t              TST_index,
										 KMNG_DES_WrappedKey_t	 TST_DESWrappedKey,
										 CRYS_DES_EncryptMode_t  TST_OpMode,
										 DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*		TST_Input_ptr;
	DxError_t	    TST_Status = DX_SUCCESS;

	/* copy input to output in case of encrypt*/
	if (TST_OpMode == CRYS_DES_Encrypt)
	{      
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];
	}
	else
	{
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput[0];
	}

	/* 1. Execute CRYS_KMNG_DES operation */
	/**************************************/
	TST_Status = CRYS_KMNG_DES(TST_DESWrappedKey,
		SEP_KeysOpSym_TestData[TST_index].TST_IV,       
		TST_OpMode, 
		SEP_KeysOpSym_TestData[TST_index].TST_DESMode,       
		TST_Input_ptr,        
		SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
		SEP_KeysOpSym_TestData[TST_index].TST_OutputData);   

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_DES",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* 2. Verify that output is as expected */
	/****************************************/
	if (TST_OpMode == CRYS_DES_Encrypt)
	{
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);
	}
	else
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_Data,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);  

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);
EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_DESNonIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_DESWrappedKey - wrapped DES key
*	TST_OpMode - operation mode
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function execute KMNG DES non integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_DES_Init operation
* 2. Execute CRYS_KMNG_DES_Block operation
* 3. Finish the operation
* 4. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_DESNonIntegrated(DxUint32_t              TST_index,
											KMNG_DES_WrappedKey_t	TST_DESWrappedKey,
											CRYS_DES_EncryptMode_t  TST_OpMode,
											DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*				TST_Input_ptr;
	DxError_t				TST_Status = DX_SUCCESS;
	CRYS_DESUserContext_t   TST_DesContext; 

	/* copy input to output in case of encrypt*/
	if (TST_OpMode == CRYS_DES_Encrypt)
	{      
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];
	}
	else
	{
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput[0];
	}

	/* 1. Execute CRYS_KMNG_DES_Init operation */
	/*******************************************/
	TST_Status = CRYS_KMNG_DES_Init(TST_DESWrappedKey,
		&TST_DesContext,
		SEP_KeysOpSym_TestData[TST_index].TST_IV,       
		TST_OpMode, 
		SEP_KeysOpSym_TestData[TST_index].TST_DESMode);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_DES_Init",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 2. Execute CRYS_KMNG_DES_Block operation */
	/*******************************************/
	TST_Status = CRYS_DES_Block(&TST_DesContext,   
		TST_Input_ptr,     
		SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
		SEP_KeysOpSym_TestData[TST_index].TST_OutputData); 

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_DES_Block",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 3. Finish the operation */
	/***************************/
	TST_Status = CRYS_DES_Free(&TST_DesContext);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"CRYS_DES_Free",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 4. Verify that output is as expected */
	/****************************************/
	if (TST_OpMode == CRYS_DES_Encrypt)
	{
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);
	}
	else
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_Data,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);  

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);
EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_AESWrappedKey - wrapped AES key
*	TST_OpMode - operation mode
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function execute KMNG AES integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_AES operation
* 2. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AESIntegrated(DxUint32_t              TST_index,
										 KMNG_AES_WrappedKey_t	 TST_AESWrappedKey,
										 CRYS_AES_EncryptMode_t  TST_OpMode,
										 DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*		TST_Input_ptr;
	DxError_t	    TST_Status = DX_SUCCESS;

	/* copy input to output in case of encrypt*/
	if (TST_OpMode == CRYS_AES_Encrypt)
	{      
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];
	}
	else
	{
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput[0];
	}

	/* 1. Execute CRYS_KMNG_AES function */
	/*************************************/
	TST_Status = CRYS_KMNG_AES(TST_AESWrappedKey,
							  SEP_KeysOpSym_TestData[TST_index].TST_IV,       
							  TST_OpMode, 
							  SEP_KeysOpSym_TestData[TST_index].TST_AESMode,       
							  TST_Input_ptr,        
							  SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
							  SEP_KeysOpSym_TestData[TST_index].TST_OutputData);   

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
							   "CRYS_KMNG_AES",
								SEP_KeysOpSym_TestData[TST_index].TST_Name,
								TST_MasterTestName);

	
	/* 2. Verify that output is as expected */
	/****************************************/
	if (TST_OpMode == CRYS_AES_Encrypt)
	{
		if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_MAC_mode)
			TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,TST_KMNG_MAC_SIZE_BYTE);
		else
			TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);
	}
	else
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_Data,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);  

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
								"DX_VOS_MemCmp",
								SEP_KeysOpSym_TestData[TST_index].TST_Name,
								TST_MasterTestName);
EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESNonIntegrated         
* 
* Inputs:
*	TST_index - test index
*	TST_AESWrappedKey - wrapped AES key
*	TST_OpMode - operation mode
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     
* Description:                                                                 
*  This function execute KMNG AES non integrated operation.
* Algorithm:       
* 1. Execute CRYS_KMNG_AES_Init operation
* 2. Execute CRYS_AES_Block operation
* 3. Finish the operation
* 4. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AESNonIntegrated(DxUint32_t              TST_index,
										    KMNG_AES_WrappedKey_t	TST_AESWrappedKey,
										    CRYS_AES_EncryptMode_t  TST_OpMode,
										    DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*				TST_Input_ptr;
	DxError_t				TST_Status = DX_SUCCESS;
	CRYS_AESUserContext_t   TST_AesContext; 

	/* copy input to output in case of encrypt*/
	if (TST_OpMode == CRYS_AES_Encrypt)
	{      
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];
	}
	else
	{
		TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput[0];
	}

	/* 1. Execute CRYS_KMNG_AES_Init operation */
	/*******************************************/
	TST_Status = CRYS_KMNG_AES_Init(TST_AESWrappedKey,
									&TST_AesContext,
									SEP_KeysOpSym_TestData[TST_index].TST_IV,       
									TST_OpMode, 
									SEP_KeysOpSym_TestData[TST_index].TST_AESMode);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
								"CRYS_KMNG_AES_Init",
								SEP_KeysOpSym_TestData[TST_index].TST_Name,
								TST_MasterTestName);

	/* 2. Execute CRYS_AES_Block operation */
	/*******************************************/
	TST_Status = CRYS_AES_Block(&TST_AesContext,   
							TST_Input_ptr,     
							SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
							SEP_KeysOpSym_TestData[TST_index].TST_OutputData); 

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
							  "CRYS_AES_Block",
							  SEP_KeysOpSym_TestData[TST_index].TST_Name,
							  TST_MasterTestName);

	/* 3. Finish the operation */
	/***************************/
	TST_Status = CRYS_AES_Finish(&TST_AesContext);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
							   "CRYS_AES_Finish",
							   SEP_KeysOpSym_TestData[TST_index].TST_Name,
							   TST_MasterTestName);

	/* 4. Verify that output is as expected */
	/****************************************/
	if (TST_OpMode == CRYS_AES_Encrypt)
	{
		if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_MAC_mode)
			TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,TST_KMNG_MAC_SIZE_BYTE);
		else
			TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_ExpectedOutput,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);
	}
	else
		TST_Status = (DxUint32_t)DX_VOS_MemCmp(SEP_KeysOpSym_TestData[TST_index].TST_Data,SEP_KeysOpSym_TestData[TST_index].TST_OutputData,SEP_KeysOpSym_TestData[TST_index].TST_DataLen);  

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);
EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESWrapUnwrap         
* 
* Inputs:
*	TST_index - test index
*	TST_AESWrappedKey - wrapped AES key
*	TST_MasterTestName - master test name   
* Outputs:
*	CRYSError_t - Function error return                                     

* Description:                                                                 
*  This function execute KMNG AES non integrated operation.
* Algorithm:       
* 1. Execute AESWrap operation
* 2. Execute AESUnwrap operation
* 3. Verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AESWrapUnwrap(DxUint32_t              TST_index,
										 KMNG_AES_WrappedKey_t	 TST_AESWrappedKey,
										 DxChar					*TST_MasterTestName) 
{
	/********* TST Variables Definitions ************/
	DxUint8_t*				TST_Input_ptr;
	DxError_t				TST_Status = DX_SUCCESS;
	DxUint32_t              TST_OutputLenTemp = SEP_KeysOpSym_TestData[TST_index].TST_OutputLen + CRYS_AES_WRAP_BLOCK_SIZE_IN_BYTES;
	DxUint8_t               TST_Output[TST_KMNG_MAX_SIZE_DATA];
	KMNG_AES_WrappedKey_t	TST_SavedAESWrappedKey;

	TST_Input_ptr = &SEP_KeysOpSym_TestData[TST_index].TST_Data[0];

	/* save AES wrapped key */
	DX_VOS_FastMemCpy(TST_SavedAESWrappedKey, TST_AESWrappedKey, KMNG_AES_WRAP_KEY_LEN);
	
	/* 1. Execute AESWrap operation */
	/********************************/
	TST_Status = CRYS_KMNG_AESWrap(TST_AESWrappedKey,      
		TST_Input_ptr,        
		SEP_KeysOpSym_TestData[TST_index].TST_DataLen,         
		SEP_KeysOpSym_TestData[TST_index].TST_OutputData,
		&TST_OutputLenTemp);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_AESWrap",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	
	/* restore AES wrapped key */
	DX_VOS_FastMemCpy(TST_AESWrappedKey, TST_SavedAESWrappedKey, KMNG_AES_WRAP_KEY_LEN);

	/* 2. Execute AESUnwrap operation */
	/********************************/
	TST_Status = CRYS_KMNG_AESUnwrap(TST_AESWrappedKey,      
		SEP_KeysOpSym_TestData[TST_index].TST_OutputData,        
		SEP_KeysOpSym_TestData[TST_index].TST_OutputLen + CRYS_AES_WRAP_BLOCK_SIZE_IN_BYTES,         
		TST_Output,
		&TST_OutputLenTemp);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,
		"CRYS_KMNG_AESUnwrap",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* 3. Verify that output is as expected */
	/****************************************/
	TST_Status = (DxUint32_t)DX_VOS_MemCmp(TST_Input_ptr,TST_Output,TST_OutputLenTemp);

	TST_ATP_CHECK_ERROR_RETURN( TST_Status,
		"DX_VOS_MemCmp",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 

}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import symmetric key
* 3. Activate AES key                                            
* 4. Call CRYS_KMNG_AES function and verify that output is as expected
* 5. Call CRYS_KMNG_AES_Init, CRYS_AES_Block, CRYS_AES_Finish functions and verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AES_ActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_AESActivateOperation";     
	KMNG_AES_WrappedKey_t	TST_AESWrappedKey;
	KMNG_AES_WrappedKey_t	TST_SavedAESWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	CRYS_AES_EncryptMode_t		TST_OpMode = CRYS_AES_Encrypt;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;
	

	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set key configuration */
	TST_userKeyConfiguration.type = SEP_KeysOpSym_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage = SEP_KeysOpSym_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpSym_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction = SEP_KeysOpSym_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[1];  	



	/* 2. Import symmetric key */
	/*********************/
	TST_Status = LLF_KMNG_ImportSymUserKey(&TST_masterKeyDataParam,
		&TST_userKeyConfiguration ,
		&TST_userKeyDataParam ,
		SEP_KeysOpSym_TestData[TST_index].TST_key);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ImportSymUserKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate AES key */
	/***********************/
	TST_Status = LLF_KMNG_ActivateAESKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		(DxUint8_t*)TST_AESWrappedKey);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateAESKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save AES wrapped key */
	DX_VOS_FastMemCpy(TST_SavedAESWrappedKey, TST_AESWrappedKey, KMNG_AES_WRAP_KEY_LEN);
	
	/* In AES MAC mode there is no decryption only encryption */
	if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_MAC_mode && TST_OpMode == CRYS_AES_Decrypt)
		TST_OpMode = CRYS_AES_Encrypt;

	/* In AES CTR mode there is only encrypt */
	if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_CTR_mode && TST_OpMode == CRYS_AES_Decrypt)
		TST_OpMode = CRYS_AES_Encrypt;

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG AES integrated \n"));

	/* 4. Call CRYS_KMNG_AES function  and verify that output is as expected */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_AESIntegrated(
							TST_index,
							TST_AESWrappedKey,
							TST_OpMode,
							TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
					"SEP_KMNG_ATP_Ext_AESIntegrated",
					SEP_KeysOpSym_TestData[TST_index].TST_Name,
					TST_MasterTestName);

	/* restore AES wrapped key */
	DX_VOS_FastMemCpy(TST_AESWrappedKey, TST_SavedAESWrappedKey, KMNG_AES_WRAP_KEY_LEN);

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG AES non integrated \n"));

	/* 5. Call CRYS_KMNG_AES_Init, CRYS_AES_Block, CRYS_AES_Finish functions */ 
	/*    and verify that output is as expected							     */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_AESNonIntegrated(
									TST_index,
									TST_AESWrappedKey,
									TST_OpMode,
									TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_AESNonIntegrated",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 


}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import symmetric key
* 3. Activate AES key                                            
* 4. Call CRYS_KMNG_AES function and verify that output is as expected
* 5. Call CRYS_KMNG_AESWrap and CRYS_KMNG_AESUnwrap functions and verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation";     
	KMNG_AES_WrappedKey_t	TST_AESWrappedKey;
	KMNG_AES_WrappedKey_t	TST_SavedAESWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	CRYS_AES_EncryptMode_t		TST_OpMode = CRYS_AES_Encrypt;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set key configuration */
	TST_userKeyConfiguration.type         = SEP_KeysOpSym_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage        = KMNG_KEY_USAGE_WRAPPING;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpSym_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction  = SEP_KeysOpSym_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[1];  	



	/* 2. Import symmetric key */
	/*********************/
	TST_Status = LLF_KMNG_ImportSymUserKey(&TST_masterKeyDataParam,
		&TST_userKeyConfiguration ,
		&TST_userKeyDataParam ,
		SEP_KeysOpSym_TestData[TST_index].TST_key);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ImportSymUserKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate AES key */
	/***********************/
	TST_Status = LLF_KMNG_ActivateAESKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		(DxUint8_t*)TST_AESWrappedKey);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateAESKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save AES wrapped key */
	DX_VOS_FastMemCpy(TST_SavedAESWrappedKey, TST_AESWrappedKey, KMNG_AES_WRAP_KEY_LEN);

	/* In AES MAC mode there is no decryption only encryption */
	if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_MAC_mode && TST_OpMode == CRYS_AES_Decrypt)
		TST_OpMode = CRYS_AES_Encrypt;

	/* In AES CTR mode there is only encrypt */
	if (SEP_KeysOpSym_TestData[TST_index].TST_AESMode == CRYS_AES_CTR_mode && TST_OpMode == CRYS_AES_Decrypt)
		TST_OpMode = CRYS_AES_Encrypt;

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG AES_Wrap and AES_Unwrap \n"));

	/* 5. Call CRYS_KMNG_AESWrap and CRYS_KMNG_AESUnwrap functions and */
	/* verify that output is as expected							   */
	/*******************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_AESWrapUnwrap(
		TST_index,
		TST_AESWrappedKey,
		TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_AESWrapUnwrap",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 


}

/************************ Public Functions ******************************/
/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_AESActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for AES key activate operation.
* Algorithm:       
* 1. Check AES operations with activated key
* 2. Check AES wrap,unwrap operations with activated key
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_AESActivateOperation( void ) 
{
	/********* Variables Definitions ************/
	DxUint32_t	SEP_Error = DX_SUCCESS;

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n ******* SEP_KMNG_ATP_Ext_AESActivateOperation test ********* \n"));

	/* 1. Check AES operations with activated key */
	/**********************************************/
	SEP_Error = SEP_KMNG_ATP_Ext_AES_ActivateOperation();

	SEP_Error = TST_CheckErrorReturn(SEP_Error,"SEP_KMNG_ATP_Ext_AESActivateOperation");

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n ******* SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation test ********* \n"));

	/* 2. Check AES wrap,unwrap operations with activated key */
	/**********************************************************/
	SEP_Error = SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation();

	SEP_Error = TST_CheckErrorReturn(SEP_Error,"SEP_KMNG_ATP_Ext_AESWrapUnwrapActivateOperation");

	return SEP_Error;
}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_DESActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for DES key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import symmetric key
* 3. Activate DES key                                            
* 4. Call CRYS_KMNG_DES function and verify that output is as expected
* 5. Call CRYS_KMNG_DES_Init, CRYS_DES_Block, CRYS_DES_Finish functions and verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_DESActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_DESActivateOperation";     
	KMNG_DES_WrappedKey_t	TST_DESWrappedKey;
	KMNG_DES_WrappedKey_t	TST_SavedDESWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	CRYS_DES_EncryptMode_t		TST_OpMode = CRYS_DES_Encrypt;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set key configuration */
	TST_userKeyConfiguration.type = SEP_KeysOpSym_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage = SEP_KeysOpSym_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpSym_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction = SEP_KeysOpSym_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[1];  	



	/* 2. Import symmetric key */
	/*********************/
	TST_Status = LLF_KMNG_ImportSymUserKey(&TST_masterKeyDataParam,
		&TST_userKeyConfiguration ,
		&TST_userKeyDataParam ,
		SEP_KeysOpSym_TestData[TST_index].TST_key);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ImportSymUserKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate DES key */
	/***********************/
	TST_Status = LLF_KMNG_ActivateDESKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		(DxUint8_t*)TST_DESWrappedKey);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateDESKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save DES wrapped key */
	DX_VOS_FastMemCpy(TST_SavedDESWrappedKey, TST_DESWrappedKey, KMNG_DES_WRAP_KEY_LEN);


	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG DES integrated \n"));

	/* 4. Call CRYS_KMNG_DES function  and verify that output is as expected */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_DESIntegrated(
		TST_index,
		TST_DESWrappedKey,
		TST_OpMode,
		TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_DESIntegrated",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* restore AES wrapped key */
	DX_VOS_FastMemCpy(TST_DESWrappedKey, TST_SavedDESWrappedKey, KMNG_DES_WRAP_KEY_LEN);

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG DES non integrated \n"));

	/* 5. Call CRYS_KMNG_DES_Init, CRYS_DES_Block, CRYS_DES_Free functions */ 
	/*    and verify that output is as expected							     */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_DESNonIntegrated(
		TST_index,
		TST_DESWrappedKey,
		TST_OpMode,
		TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_DESNonIntegrated",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 


}

/***********************************************************************
* Function Name:                                                           
*  SEP_KMNG_ATP_Ext_HMACActivateOperation         
* 
* Inputs:
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:                                                                 
*  This function Is a Acceptance test for HMAC key activate operation.
* Algorithm:       
* 1. Create a master key
* 2. Import symmetric key
* 3. Activate HMAC key                                            
* 4. Call CRYS_KMNG_HMAC function and verify that output is as expected
* 5. Call CRYS_KMNG_HMAC_Init, CRYS_HMAC_Update, CRYS_HMAC_Finish functions and verify that output is as expected
**************************************************************************************/                                
DxError_t SEP_KMNG_ATP_Ext_HMACActivateOperation( void ) 
{

	/********* TST Variables Definitions ************/
	DxUint32_t				TST_index = 0; 
	DxError_t				TST_Status = DX_SUCCESS; 
	DxUint8_t				TST_KeyRingPwd[TST_KMNG_PWD_MAX_LEN] = {0x0,0x1,0x2,0x3,0x4,0x5,0x6,0x7,0x8,0x9};
	DxUint32_t				TST_KeyRingPwdLen = 10;
	DxChar					TST_MasterTestName[TST_FUNC_STRING_LEN] = "SEP_KMNG_ATP_Ext_HMACActivateOperation";     
	KMNG_HMAC_WrappedKey_t	TST_HMACWrappedKey;
	KMNG_HMAC_WrappedKey_t	TST_SavedHMACWrappedKey;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_masterKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_masterKeyConfiguration;

	/* paramaters for create key */
	KMNG_KeyDataParameter_t		TST_userKeyDataParam;

	/* configuration parameters */
	KMNG_KeyConfiguration_t		TST_userKeyConfiguration;

	// for master key
	DxUint8_t					TST_encryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint8_t					TST_userEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];

	/* buffer to save encrypted key */
	DxUint8_t					TST_savedEncryptedKeyBuffer[KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES];
	DxUint32_t					TST_savedEncryptedKeyDataLength;


	/* set master key parameters (should be AES key) */
	TST_masterKeyDataParam.keyPassword_ptr = TST_KeyRingPwd;
	TST_masterKeyDataParam.keyPasswordLen  = TST_KeyRingPwdLen;
	TST_masterKeyDataParam.encryptedKeyData_ptr   = (DxUint8_t*)TST_encryptedKeyBuffer;
	TST_masterKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set master key configuration */
	TST_masterKeyConfiguration.type = KMNG_KeyTypeAES; 
	TST_masterKeyConfiguration.keySizeBytes = KMNG_AES_MAX_KEY_SIZE_IN_BYTES; /*256 Bits Key*/
	TST_masterKeyConfiguration.usage = KMNG_KEY_USAGE_STORAGE;
	TST_masterKeyConfiguration.restriction = KMNG_KEY_RESTRICTION_NON_EXPORTABLE;  /* instead PROTECTED.  May be other?? */

	/* 1. Create a master key */
	/**************************/
	TST_Status = LLF_KMNG_CreateKeyRingKey(&TST_masterKeyDataParam , &TST_masterKeyConfiguration);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_CreateKeyRingKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* save master key, because it will be unwrapped in each KMNG operation */
	DX_VOS_FastMemCpy(TST_savedEncryptedKeyBuffer, TST_masterKeyDataParam.encryptedKeyData_ptr, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_savedEncryptedKeyDataLength = TST_masterKeyDataParam.encryptedKeyDataLength;

	/* set user key parameters */
	TST_userKeyDataParam.keyPassword_ptr = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwd;
	TST_userKeyDataParam.keyPasswordLen = SEP_KeysOpSym_TestData[TST_index].TST_KeyPwdLen;
	/* TST_userKeyDataParam.keyId - will return from LLF layer */
	TST_userKeyDataParam.encryptedKeyData_ptr = (DxUint8_t*)TST_userEncryptedKeyBuffer;
	TST_userKeyDataParam.encryptedKeyDataLength = KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES;

	/* set key configuration */
	TST_userKeyConfiguration.type = SEP_KeysOpSym_TestData[TST_index].TST_KeyType;
	TST_userKeyConfiguration.usage = SEP_KeysOpSym_TestData[TST_index].TST_KeyUsage;
	TST_userKeyConfiguration.keySizeBytes = SEP_KeysOpSym_TestData[TST_index].TST_KeySize;  
	TST_userKeyConfiguration.restriction = SEP_KeysOpSym_TestData[TST_index].TST_KeyRestriction;

	/* if UserSpecificKeyData is not NULL, then copy user specific data into configuration structure,
	else set it to 0  */
	TST_userKeyConfiguration.UserSpecificKeyData[0] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[0];
	TST_userKeyConfiguration.UserSpecificKeyData[1] = SEP_KeysOpSym_TestData[TST_index].TST_UserData[1];  	



	/* 2. Import symmetric key */
	/*********************/
	TST_Status = LLF_KMNG_ImportSymUserKey(&TST_masterKeyDataParam,
		&TST_userKeyConfiguration ,
		&TST_userKeyDataParam ,
		SEP_KeysOpSym_TestData[TST_index].TST_key);


	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ImportSymUserKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);


	/* restore the encrypted key: */
	DX_VOS_FastMemCpy(TST_masterKeyDataParam.encryptedKeyData_ptr, TST_savedEncryptedKeyBuffer, KMNG_SYM_ENCR_KEY_BUFF_SIZE_IN_BYTES);
	TST_masterKeyDataParam.encryptedKeyDataLength = TST_savedEncryptedKeyDataLength;

	/* 3. Activate HMAC key */
	/***********************/
	TST_Status = LLF_KMNG_ActivateHMACKey(&TST_masterKeyDataParam,
		&TST_userKeyDataParam,
		(DxUint8_t*)TST_HMACWrappedKey);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"LLF_KMNG_ActivateHMACKey",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* save HMAC wrapped key */
	DX_VOS_FastMemCpy(TST_SavedHMACWrappedKey, TST_HMACWrappedKey, KMNG_HMAC_WRAP_KEY_LEN);


	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG HMAC integrated \n"));

	/* 4. Call CRYS_KMNG_HMAC function  and verify that output is as expected */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_HMACIntegrated(
		TST_index,
		TST_HMACWrappedKey,
		TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_HMACIntegrated",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

	/* restore HMAC wrapped key */
	DX_VOS_FastMemCpy(TST_HMACWrappedKey, TST_SavedHMACWrappedKey, KMNG_HMAC_WRAP_KEY_LEN);

	ATP_LOG_TST_PRINT((KMNG_ATP_TST_MSG,"\n KMNG HMAC non integrated \n"));

	/* 5. Call CRYS_KMNG_HMAC_Init, CRYS_HMAC_Update, CRYS_HMAC_Finish functions */ 
	/*    and verify that output is as expected							     */
	/*************************************************************************/
	TST_Status = SEP_KMNG_ATP_Ext_HMACNonIntegrated(
		TST_index,
		TST_HMACWrappedKey,
		TST_MasterTestName);

	TST_ATP_CHECK_ERROR_RETURN(TST_Status,
		"SEP_KMNG_ATP_Ext_HMACNonIntegrated",
		SEP_KeysOpSym_TestData[TST_index].TST_Name,
		TST_MasterTestName);

EXIT_ON_ERROR:
	return TST_Status; 

}




